const { Plugin, setIcon, SuggestModal, getAllTags } = require("obsidian");

class TagSearchPlugin extends Plugin {
    async onload() {
        this.addCommand({
            id: "open-tag-search",
            name: "Open tag search",
            callback: () => this.openTagSearch()
        });

        this.registerEvent(
            this.app.workspace.on("active-leaf-change", () => {
                this.addHeaderButton();
            })
        );

        this.addHeaderButton();
    }

    addHeaderButton() {
        const leaf = this.app.workspace.activeLeaf;
        if (!leaf || !leaf.view || !leaf.view.headerEl) return;

        const headerEl = leaf.view.headerEl;
        if (!headerEl) return;

        if (headerEl.querySelector(".tag-search-button")) return;

        const actionsEl = headerEl.querySelector(".view-actions");
        if (!actionsEl) return;

        const btn = document.createElement("div");
        btn.classList.add("view-action", "tag-search-button");
        btn.style.display = "flex";
        btn.style.alignItems = "center";
        btn.style.justifyContent = "center";

        setIcon(btn, "hashtag");
        btn.setAttribute("aria-label", "Apri ricerca tag");

        btn.addEventListener("click", () => this.openTagSearch());

        const moreOptionsBtn = actionsEl.querySelector(".view-action:last-child");
        if (moreOptionsBtn) {
            actionsEl.insertBefore(btn, moreOptionsBtn);
        } else {
            actionsEl.appendChild(btn);
        }
    }

    openTagSearch() {
        new NoteByTagModal(this.app).open();
    }

    onunload() {
        document.querySelectorAll(".tag-search-button").forEach(el => el.remove());
    }
}

class NoteByTagModal extends SuggestModal {
    constructor(app) {
        super(app);
        this.app = app;
        this.setPlaceholder("Cerca note per tag…");
    }

    getSuggestions(query) {
        const q = (query || "").trim().toLowerCase();
        if (!q) return [];

        const files = this.app.vault.getMarkdownFiles();
        const matches = [];

        for (const file of files) {
            const cache = this.app.metadataCache.getCache(file.path);
            if (!cache) continue;

            const tags = (getAllTags(cache) || []).map(t => t.replace(/^#/, "").toLowerCase());
            if (tags.length === 0) continue;

            let bestScore = 0;
            for (const tag of tags) {
                const s = this.fuzzyScore(tag, q);
                if (s > bestScore) bestScore = s;
                if (bestScore === 1) break; // match perfetto
            }

            if (bestScore > 0) {
                matches.push({ file, score: bestScore });
            }
        }

        matches.sort((a, b) => {
            if (b.score !== a.score) return b.score - a.score;
            return a.file.basename.localeCompare(b.file.basename);
        });

        return matches;
    }

    renderSuggestion(item, el) {
        el.createDiv({ text: item.file.basename, cls: "tag-search-title" });
    }

    onChooseSuggestion(item, evt) {
        const leaf = this.app.workspace.getLeaf(true);
        leaf.openFile(item.file, { active: true });
        this.close();
    }

    fuzzyScore(text, pattern) {
        if (!pattern) return 0;
        if (text === pattern) return 1;
        if (text.startsWith(pattern)) return 0.95;
        if (text.includes(pattern)) return 0.85;

        let ti = 0;
        for (let pi = 0; pi < pattern.length; pi++) {
            const ch = pattern[pi];
            ti = text.indexOf(ch, ti);
            if (ti === -1) return 0;
            ti++;
        }
        return 0.7;
    }
}

module.exports = TagSearchPlugin;